<?php
/**
 * Plugin Name: Markdown to HTML on Publish (v1.7)
 * Plugin URI: https://gptchronicle.com
 * Description: Converts Markdown to HTML only when transitioning to 'publish'. Supports all common markdown features.
 * Version: 1.7
 * Author: GptChronicle.com
 * Author URI: https://gptchronicle.com
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

if (!defined('ABSPATH')) exit;

/*
DISCLAIMER:
This plugin is provided "as is" without warranties or guarantees of functionality.
GptChronicle.com assumes no responsibility for misuse, site disruptions, or data loss. Use at your own risk.
*/

// Set default option to off on activation
register_activation_hook(__FILE__, function () {
    if (get_option('markdown_enabled') === false) {
        update_option('markdown_enabled', '0');
    }
});

// Admin toggle setting
add_action('admin_menu', function () {
    add_options_page('Markdown Settings', 'Markdown Settings', 'manage_options', 'markdown-settings', function () {
        ?>
        <div class="wrap">
            <h1>Markdown Conversion Settings</h1>
            <form method="post" action="options.php">
                <?php
                settings_fields('markdown_settings_group');
                do_settings_sections('markdown_settings_group');
                ?>
                <table class="form-table">
                    <tr valign="top">
                        <th scope="row">Enable Markdown Conversion</th>
                        <td>
                            <input type="checkbox" name="markdown_enabled" value="1" <?php checked(get_option('markdown_enabled'), '1'); ?> />
                            <label>Convert Markdown to HTML only when publishing posts or pages.</label>
                        </td>
                    </tr>
                </table>
                <?php submit_button(); ?>
            </form>
        </div>
        <?php
    });
});

add_action('admin_init', function () {
    register_setting('markdown_settings_group', 'markdown_enabled');
});

// Inline markdown parsing
function parse_inline_markdown($text) {
    $text = preg_replace('/~~(.*?)~~/', '<del>$1</del>', $text);
    $text = preg_replace('/\*\*(.*?)\*\*/', '<strong>$1</strong>', $text);
    $text = preg_replace('/(?<!\*)\*(?!\*)(.*?)\*(?!\*)/', '<em>$1</em>', $text);
    $text = preg_replace('/!\[(.*?)\]\((.*?)\)/', '<img src="$2" alt="$1" />', $text);
    $text = preg_replace('/\[(.*?)\]\((.*?)\)/', '<a href="$2">$1</a>', $text);
    return $text;
}

// Core Markdown conversion
function convert_markdown_to_html($raw) {
    $lines = preg_split("/\r?\n/", $raw);
    $html = "";
    $in_ul = false;
    $in_ol = false;
    $in_pre = false;
    $table_started = false;

    foreach ($lines as $line) {
        $line = trim($line);
        if ($line === '') {
            if ($in_ul) { $html .= "</ul>\n"; $in_ul = false; }
            if ($in_ol) { $html .= "</ol>\n"; $in_ol = false; }
            if ($table_started) { $html .= "</table>\n"; $table_started = false; }
            continue;
        }

        if (preg_match('/^```/', $line)) {
            if (!$in_pre) {
                $html .= "<pre><code>";
                $in_pre = true;
            } else {
                $html .= "</code></pre>\n";
                $in_pre = false;
            }
            continue;
        }
        if ($in_pre) {
            $html .= htmlspecialchars($line) . "\n";
            continue;
        }

        if (preg_match('/^> (.*)/', $line, $m)) {
            $html .= "<blockquote>" . parse_inline_markdown($m[1]) . "</blockquote>\n";
            continue;
        }

        if (preg_match('/^### (.*)/', $line, $m)) {
            $html .= "<h3>" . parse_inline_markdown($m[1]) . "</h3>\n"; continue;
        }
        if (preg_match('/^## (.*)/', $line, $m)) {
            $html .= "<h2>" . parse_inline_markdown($m[1]) . "</h2>\n"; continue;
        }
        if (preg_match('/^# (.*)/', $line, $m)) {
            $html .= "<h1>" . parse_inline_markdown($m[1]) . "</h1>\n"; continue;
        }

        if (preg_match('/^[-*] (.*)/', $line, $m)) {
            if (!$in_ul) { $html .= "<ul>\n"; $in_ul = true; }
            $html .= "<li>" . parse_inline_markdown($m[1]) . "</li>\n";
            continue;
        } elseif ($in_ul) {
            $html .= "</ul>\n"; $in_ul = false;
        }

        if (preg_match('/^\d+\. (.*)/', $line, $m)) {
            if (!$in_ol) { $html .= "<ol>\n"; $in_ol = true; }
            $html .= "<li>" . parse_inline_markdown($m[1]) . "</li>\n";
            continue;
        } elseif ($in_ol) {
            $html .= "</ol>\n"; $in_ol = false;
        }

        if (preg_match('/^\|(.+)\|$/', $line)) {
            $cols = array_map('trim', explode('|', trim($line, '|')));
            if (!$table_started) {
                $html .= "<table><tr>";
                foreach ($cols as $col) $html .= "<th>" . parse_inline_markdown($col) . "</th>";
                $html .= "</tr>\n";
                $table_started = true;
            } else {
                $html .= "<tr>";
                foreach ($cols as $col) $html .= "<td>" . parse_inline_markdown($col) . "</td>";
                $html .= "</tr>\n";
            }
            continue;
        }

        $html .= "<p>" . parse_inline_markdown($line) . "</p>\n";
    }

    if ($in_ul) { $html .= "</ul>\n"; }
    if ($in_ol) { $html .= "</ol>\n"; }
    if ($table_started) { $html .= "</table>\n"; }
    if ($in_pre) { $html .= "</code></pre>\n"; }

    return $html;
}

// Hook into publish transition only
add_action('transition_post_status', function ($new_status, $old_status, $post) {
    if (get_option('markdown_enabled') !== '1') return;
    if ($old_status === 'publish' || $new_status !== 'publish') return;
    if (!in_array($post->post_type, ['post', 'page'])) return;

    $html = convert_markdown_to_html($post->post_content);
    remove_action('save_post', 'markdown_to_html_on_publish'); // prevent loops
    wp_update_post([
        'ID' => $post->ID,
        'post_content' => $html
    ]);
}, 10, 3);
?>
